/**************************************************************************
  This is an example for our Monochrome OLEDs based on SSD1306 drivers

  Pick one up today in the adafruit shop!
  ------> http://www.adafruit.com/category/63_98

  This example is for a 128x32 pixel display using I2C to communicate
  3 pins are required to interface (two I2C and one reset).

  Adafruit invests time and resources providing this open
  source code, please support Adafruit and open-source
  hardware by purchasing products from Adafruit!

  Written by Limor Fried/Ladyada for Adafruit Industries,
  with contributions from the open source community.
  BSD license, check license.txt for more information
  All text above, and the splash screen below must be
  included in any redistribution.
 **************************************************************************/

// == インクルード 必要ファイルの読み込み ==
#include <SPI.h>
#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>

#define SCREEN_WIDTH 128    // 横の画素数
#define SCREEN_HEIGHT 64    // 縦の画素数
#define OLED_RESET     4    // ピン番号のリセット (Arduinoリセットピンを共有しているなら -1 )
#define SCREEN_ADDRESS 0x3C // アドレス

Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);

#define NUMFLAKES     10 // 雪の欠片の数
#define LOGO_HEIGHT   16
#define LOGO_WIDTH    16
static const unsigned char PROGMEM logo_bmp[] =   // 配列
{ 0b00000000, 0b11000000,
  0b00000001, 0b11000000,
  0b00000001, 0b11000000,
  0b00000011, 0b11100000,
  0b11110011, 0b11100000,
  0b11111110, 0b11111000,
  0b01111110, 0b11111111,
  0b00110011, 0b10011111,
  0b00011111, 0b11111100,
  0b00001101, 0b01110000,
  0b00011011, 0b10100000,
  0b00111111, 0b11100000,
  0b00111111, 0b11110000,
  0b01111100, 0b11110000,
  0b01110000, 0b01110000,
  0b00000000, 0b00110000
};

void setup() {
  // シリアル通信
  Serial.begin(9600);

  // 初期化
  if (!display.begin(SSD1306_SWITCHCAPVCC, SCREEN_ADDRESS)) {
    Serial.println(F("SSD1306 allocation failed"));
    for (;;); // ループ
  }

  // Adafruit のロゴが2秒間表示される　（バッファにあるもの）
  display.display();
  delay(2000);

  // バッファを空にする
  display.clearDisplay();

  // 表示する内容をバッファへと書き込む
  display.drawPixel(10, 10, SSD1306_WHITE);

  // 画面を更新 -- 2秒間
  display.display();
  delay(2000);

  testdrawline();      //  たくさんの線を描画

  testdrawrect();      //  四角を描画 (アウトライン)

  testfillrect();      //  四角を描画 （塗りつぶし）

  testdrawcircle();    // 円を描画（アウトライン）

  testfillcircle();    // 円を描画(塗りつぶし)

  testdrawroundrect(); // 角丸の長方形を描画（アウトライン）

  testfillroundrect(); // 角丸の長方形を描画(塗りつぶし)

  testdrawtriangle();  // 三角形を描画(アウトライン)

  testfilltriangle();  // 三角形を描画(塗りつぶし)

  testdrawchar();      // デフォルトのフォントの文字を描画

  testdrawstyles();    // 様式化された文字を描画

  testscrolltext();    // スクロールテキストを描画

  testdrawbitmap();    // 小さなビットマップ画像を描画


  display.invertDisplay(true); // 白 (1) - 黒(0)が逆転
  delay(1000);                 // 1秒間停止
  display.invertDisplay(false);// 反転を戻す
  delay(1000);                 // 1秒間停止

  // ビットマップをアニメーション化
  testanimate(logo_bmp, LOGO_WIDTH, LOGO_HEIGHT);
}

void loop() {
}

void testdrawline() {  // たくさんの線を描画
  int16_t i;
  display.clearDisplay(); //  バッファを空にする

  /*
     display.drawLine()

      線を描画します
      display.drawLine(x座標の開始点, y座標の開始点, x座標の終了点, y座標の終了点, 線の色)
  */

  //  ==  ディスプレイの横幅までループ ==
  for (i = 0; i < display.width(); i += 4) {
    display.drawLine(0, 0, i, display.height() - 1, SSD1306_WHITE); //  線を描画
    display.display(); // 画面を更新
    delay(1); // 1ミリ秒停止
  }
  //  |||  ディスプレイの縦幅までループ |||
  for (i = 0; i < display.height(); i += 4) {
    display.drawLine(0, 0, display.width() - 1, i, SSD1306_WHITE); // 線を描画
    display.display();   // 画面を更新
    delay(1);   // 1ミリ秒停止
  }
  delay(250);  // 250ミリ秒停止

  display.clearDisplay();  // バッファを空にする

  //　==　ディスプレイの横幅までループ ==
  for (i = 0; i < display.width(); i += 4) {
    display.drawLine(0, display.height() - 1, i, 0, SSD1306_WHITE); // 線を描画
    display.display();  // 画面を更新
    delay(1); // 1ミリ秒停止
  }

  //  |||  ディスプレイの縦幅までループ  |||
  for (i = display.height() - 1; i >= 0; i -= 4) {
    display.drawLine(0, display.height() - 1, display.width() - 1, i, SSD1306_WHITE); // 線を描画
    display.display(); // 画面を更新
    delay(1);   //  1ミリ秒停止
  }
  delay(250);  // 250ミリ秒停止


  display.clearDisplay();  //  バッファを空にする


  //  ==  ディスプレイの横幅までループ ==
  for (i = display.width() - 1; i >= 0; i -= 4) {
    display.drawLine(display.width() - 1, display.height() - 1, i, 0, SSD1306_WHITE); //  線を描画
    display.display();  //  画面を更新
    delay(1);  // 1ミリ秒停止
  }

  // |||  ディスプレイの縦幅までループ  |||
  for (i = display.height() - 1; i >= 0; i -= 4) {
    display.drawLine(display.width() - 1, display.height() - 1, 0, i, SSD1306_WHITE); //  線を描画
    display.display();    //   画面を更新
    delay(1);     // 　　　１ミリ秒停止
  }


  delay(250);     //   250ミリ秒間停止


  display.clearDisplay();     //　　バッファを空にする

  //  |||  ディスプレイの縦幅までループ  |||
  for (i = 0; i < display.height(); i += 4) {
    display.drawLine(display.width() - 1, 0, 0, i, SSD1306_WHITE); //  線を描画
    display.display();  // 画面を更新
    delay(1);   //  1ミリ秒停止
  }
  //  ==  ディスプレイの横幅までループ  ===
  for (i = 0; i < display.width(); i += 4) {
    display.drawLine(display.width() - 1, 0, i, display.height() - 1, SSD1306_WHITE); // 　線を描画
    display.display();   //   画面を更新
    delay(1);   // 1ミリ秒停止
  }

  delay(2000); //   2秒間停止
}



void testdrawrect(void) {
  display.clearDisplay(); //  バッファを空にする

  /*
      display.drawRect()

      四角を描画します  (アウトライン)
      display.drawRect(x座標のの開始点, y座標の開始点, 幅, 高さ, 線の色)
  */

  //  == ディスプレイの縦幅 / 2 まで  ループ ==
  for (int16_t i = 0; i < display.height() / 2; i += 2) {
    display.drawRect(i, i, display.width() - 2 * i, display.height() - 2 * i, SSD1306_WHITE); // ループするたびに内側へ
    display.display(); //  画面を更新
    delay(1);  // 1ミリ秒停止
  }

  delay(2000);   //　　2秒間停止
}

void testfillrect(void) {
  display.clearDisplay();   //   バッファを空にする1

  /*
     display.fillRect()

     四角を描画　（塗りつぶしあり）
     display.ffillRect(x座標のの開始点, y座標の開始点, 幅, 高さ, 線の色)
  */

  //  == ディスプレイの縦幅 / 2  までループ ==
  for (int16_t i = 0; i < display.height() / 2; i += 3) {
    display.fillRect(i, i, display.width() - i * 2, display.height() - i * 2, SSD1306_INVERSE); //  四角を描画
    display.display(); // 画面を更新
    delay(1); // 1ミリ秒停止
  }

  delay(2000); // 2秒間停止
}

void testdrawcircle(void) {
  display.clearDisplay();//  バッファを空にする

  /*
      display.drawCircle()

     円を描画 (アウトライン)
     display.drawCircle(円の中心のx座標, 円の中心のy座標, 円の半径, 線の色)
  */

  //  ==  (横幅,縦幅)の大きい方の値　/ 2　　まで　ループ ==
  for (int16_t i = 0; i < max(display.width(), display.height()) / 2; i += 2) {
    display.drawCircle(display.width() / 2, display.height() / 2, i, SSD1306_WHITE); //  円を描画
    display.display();  //  画面を更新
    delay(1);  //  1ミリ秒停止
  }

  delay(2000);   //  2秒間停止
}

void testfillcircle(void) {
  display.clearDisplay();  //  バッファを空にする

  /*
     display.fillCircle()

     円を描画　（塗りつぶしあり）
     display.fillCircle(円の中心x座標, 円の中心y座標, 円の半径, 線の色)
  */

  // ループ
  for (int16_t i = max(display.width(), display.height()) / 2; i > 0; i -= 3) {
    // INVERSEカラーが使用されるため、円は白/黒を交互に表示します
    display.fillCircle(display.width() / 2, display.height() / 2, i, SSD1306_INVERSE);
    display.display(); // 画像を更新
    delay(1);  // 1ミリ秒停止
  }

  delay(2000);   // 2秒停止
}

void testdrawroundrect(void) {
  display.clearDisplay();  //バッファを空にする

  /*
       display.drawRountRect()

      四角を描画(塗りつぶしなし)
      display.drawRect(左上x, 左上y, 幅, 高さ, 線の色)
  */

  // ループ
  for (int16_t i = 0; i < display.height() / 2 - 2; i += 2) {
    display.drawRoundRect(i, i, display.width() - 2 * i, display.height() - 2 * i,
                          display.height() / 4, SSD1306_WHITE);
    display.display();  // 画面を更新
    delay(1);  // 1ミリ秒停止
  }

  delay(2000);  // 2秒間停止
}

void testfillroundrect(void) {
  display.clearDisplay();  // バッファを空にする

  for (int16_t i = 0; i < display.height() / 2 - 2; i += 2) {
    //  角丸の四角を描画
    display.fillRoundRect(i, i, display.width() - 2 * i, display.height() - 2 * i,
                          display.height() / 4, SSD1306_INVERSE);
    display.display();
    delay(1);
  }

  delay(2000);
}

void testdrawtriangle(void) {
  display.clearDisplay();  //  バッファを空にする
  //  ループ
  for (int16_t i = 0; i < max(display.width(), display.height()) / 2; i += 5) {
    // 三角形を描画
    display.drawTriangle(
      display.width() / 2  , display.height() / 2 - i,
      display.width() / 2 - i, display.height() / 2 + i,
      display.width() / 2 + i, display.height() / 2 + i, SSD1306_WHITE);
    display.display();
    delay(1);
  }

  delay(2000);
}

void testfilltriangle(void) {
  display.clearDisplay();   //  バッファを空にする

  for (int16_t i = max(display.width(), display.height()) / 2; i > 0; i -= 5) {
    // T三角形を描画（塗りつぶしあり）
    display.fillTriangle(
      display.width() / 2  , display.height() / 2 - i,
      display.width() / 2 - i, display.height() / 2 + i,
      display.width() / 2 + i, display.height() / 2 + i, SSD1306_INVERSE);
    display.display();
    delay(1);
  }

  delay(2000); // 2秒間停止
}

void testdrawchar(void) {
  display.clearDisplay();   //   バッファを空にする

  display.setTextSize(1);      // 文字サイズの設定
  display.setTextColor(SSD1306_WHITE); // 文字の色の設定
  display.setCursor(0, 0);     // カーソルの位置の設定
  display.cp437(true);         // Use full 256 char 'Code Page 437' font

  //  ループ
  for (int16_t i = 0; i < 256; i++) {
    if (i == '\n') display.write(' ');
    else          display.write(i);
  }

  display.display();   //  画面の更新
  delay(2000);    //  2秒間停止
}

void testdrawstyles(void) {
  display.clearDisplay();    //  バッファを空にする

  display.setTextSize(1);             // 　テキストサイズの設定
  display.setTextColor(SSD1306_WHITE);        // 文字色の設定
  display.setCursor(0, 0);            // カーソル位置の設定
  display.println(F("Hello, world!")); // Flush ram から直接読む

  display.setTextColor(SSD1306_BLACK, SSD1306_WHITE); // 色の設定
  display.println(3.141592); // 文字の出力

  display.setTextSize(2);             // 文字サイズの設定
  display.setTextColor(SSD1306_WHITE);   // 文字の色の設定
  display.print(F("0x"));       //  文字の出力
  display.println(0xDEADBEEF, HEX);   //  文字の出力

  display.display();  //  画面の更新
  delay(2000);  //  2秒間停止
}

void testscrolltext(void) {
  display.clearDisplay();  //  バッファを空にする

  display.setTextSize(2); // テキストサイズの設定
  display.setTextColor(SSD1306_WHITE);  //  文字の色
  display.setCursor(10, 0);  //  カーソルの位置
  display.println(F("scroll"));  // Flush ram から直接読む
  display.display();      // 画面の更新
  delay(100);     // 100ミリ秒停止

  //  ==  スクロール  ==
  display.startscrollright(0x00, 0x0F);  // 右スクロール
  delay(2000);
  display.stopscroll();    //  スクロールの停止
  delay(1000);
  display.startscrollleft(0x00, 0x0F);  //  左スクロール
  delay(2000);
  display.stopscroll(); // スクロールの停止
  delay(1000);
  display.startscrolldiagleft(0x00, 0x07);  // 上へスクロール
  delay(2000);
  display.startscrolldiagright(0x00, 0x07); //  下へスクロール
  delay(2000);
  display.stopscroll();  //  スクロールの停止
  delay(1000);
  display.display();      // 画面の更新
}

void testdrawbitmap(void) {
  display.clearDisplay();  //  バッファを空にする

  display.drawBitmap(
    (display.width()  - LOGO_WIDTH ) ,
    (display.height() - LOGO_HEIGHT),
    logo_bmp, LOGO_WIDTH, LOGO_HEIGHT, 1);
  display.display();
  delay(1000);
}

#define XPOS   0
#define YPOS   1
#define DELTAY 2

void testanimate(const uint8_t *bitmap, uint8_t w, uint8_t h) {
  int8_t f, icons[NUMFLAKES][3];
  display.clearDisplay();   //   バッファを空にする

  //  雪の位置を初期化
  for (f = 0; f < NUMFLAKES; f++) {
    icons[f][XPOS]   = random(1 - LOGO_WIDTH, display.width());
    icons[f][YPOS]   = -LOGO_HEIGHT;
    icons[f][DELTAY] = random(1, 6);
    Serial.print(F("x: "));
    Serial.print(icons[f][XPOS], DEC);
    Serial.print(F(" y: "));
    Serial.print(icons[f][YPOS], DEC);
    Serial.print(F(" dy: "));
    Serial.println(icons[f][DELTAY], DEC);
  }

  for (;;) { //  ループ
    display.clearDisplay(); // バッファを空にする
    //  ループ
    for (f = 0; f < NUMFLAKES; f++) {
      display.drawBitmap(icons[f][XPOS], icons[f][YPOS], bitmap, w, h, SSD1306_WHITE);
    }

    display.display(); // 画面の更新
    delay(200);        // 200ミリ秒停止

    /*
        display.drawBitmap(x,y,bitmap,rotation,flip)

       画面上にバイト配列であるビットマップを表示します
       display.drawBitmap(x座標,y座標,bitmap,オプション(ビットマップの回転),オプション(ビットマップの反転))
                                            ↑NOROT, ROTCCW, ROT180, ROTCW   ↑NOFLIP, FLIPH, FLIPV, FLIPVH
    */

    // 雪の描画
    //  ループ
    for (f = 0; f < NUMFLAKES; f++) {
      icons[f][YPOS] += icons[f][DELTAY];
      //  雪が画面の下部から外れている場合
      if (icons[f][YPOS] >= display.height()) {
        //上部から少し離れたランダムな位置に再初期化します
        icons[f][XPOS]   = random(1 - LOGO_WIDTH, display.width());
        icons[f][YPOS]   = -LOGO_HEIGHT;
        icons[f][DELTAY] = random(1, 6);
      }
    }
  }
}
